<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DocumentoPessoa;
use App\Models\Lote;
use App\Models\User;
use App\Repositories\Contracts\UserRepositoryInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Validator;

class UserController extends Controller
{
    private $route = 'users';
    private $paginate = 10;
    private $search = ['nome', 'email', 'sobrenome'];
    private $model;

    public function __construct(UserRepositoryInterface $model)
    {
        $this->model = $model;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $columnList = [
            'id' => '#', 'nome' => trans('agendamento.name'), 'email' => trans('agendamento.email'),
            'cpf' => trans('agendamento.cpf'), 'sobrenome' => trans('agendamento.sobrenome'), 'telefone' => trans('agendamento.telefone'),
            'data_nascimento' => trans('agendamento.data_nascimento'),
        ];

        $page = trans('agendamento.user_list');

        $search = "";
        if (isset($request->search)) {
            $search = $request->search;
            $list = $this->model->findWhereLike($this->search, $search, 'id', 'DESC');
        } else {
            $list = $this->model->paginate($this->paginate);
        }

        $routeName = $this->route;

        $breadcrumb = [
            (object) ['url' => route('home'), 'title' => trans('agendamento.home')],
            (object) ['url' => '', 'title' => trans('agendamento.list', ['page' => $page])],
        ];

        return view('admin.' . $routeName . '.index', [
            'list' => $list, 'search' => $search,
            'page' => $page, 'routeName' => $routeName, 'columnList' => $columnList, 'breadcrumb' => $breadcrumb,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $routeName = $this->route;
        $page = trans('agendamento.user_list');
        $page_create = trans('agendamento.user');
        //$roles = $this->modelRole->all('name','ASC');

        $breadcrumb = [
            (object) ['url' => route('home'), 'title' => trans('agendamento.home')],
            (object) ['url' => route($routeName . '.index'), 'title' => trans('agendamento.list', ['page' => $page])],
            (object) ['url' => '', 'title' => trans('agendamento.create_crud', ['page' => $page_create])],
        ];

        return view('admin.' . $routeName . '.create', ['page' => $page, 'page_create' => $page_create, 'routeName' => $routeName, 'breadcrumb' => $breadcrumb]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $data = $request->all();
        $data['password'] = Hash::make($data['password']);

        Validator::make($data, [
            'nome' => 'required|string|max:255',
            'sobrenome' => 'required|string',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'cpf' => 'required|string|min:11',
            'telefone' => 'required|string|min:11',
            'data_nascimento' => 'required',
        ]);

        $documentosPessoa = Lote::where('cpf', $data['cpf'])->first();

        if ($documentosPessoa) {
            $this->model->create($data);
            // session()->flash('msg', trans('agendamento.record_added_successfully'));
            // session()->flash('status', 'success'); // success error notification
            return redirect('cadastroConfirmacao');
        } elseif ($documentosPessoa == null) {
            session()->flash('msg', trans('agendamento.error_adding_registry'));
            session()->flash('status', 'error'); // success error notification
            return redirect()->back();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id, Request $request)
    {
        $routeName = $this->route;

        $register = $this->model->find($id);

        if ($register) {
            $page = trans('agendamento.user_list');
            $page2 = trans('agendamento.user');

            $breadcrumb = [
                (object) ['url' => route('home'), 'title' => trans('agendamento.home')],
                (object) ['url' => route($routeName . '.index'), 'title' => trans('agendamento.list', ['page' => $page])],
                (object) ['url' => '', 'title' => trans('agendamento.show_crud', ['page' => $page2])],
            ];
            $delete = false;
            if ($request->delete ?? false) {
                session()->flash('msg', trans('agendamento.delete_this_record'));
                session()->flash('status', 'notification'); // success error notification
                $delete = true;
            }

            return view('admin.' . $routeName . '.show', ['page' => $page, 'delete' => $delete, 'page2' => $page2, 'register' => $register, 'routeName' => $routeName, 'breadcrumb' => $breadcrumb]);
        }
        return redirect()->route($routeName . '.index');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $routeName = $this->route;

        $register = $this->model->find($id);

        if ($register) {
            $page = trans('agendamento.user_list');
            $page2 = trans('agendamento.user');

            $breadcrumb = [
                (object) ['url' => route('home'), 'title' => trans('agendamento.home')],
                (object) ['url' => route($routeName . '.index'), 'title' => trans('agendamento.list', ['page' => $page])],
                (object) ['url' => '', 'title' => trans('agendamento.edit_crud', ['page' => $page2])],
            ];

            return view('admin.' . $routeName . '.edit', ['page' => $page, 'page2' => $page2, 'register' => $register, 'routeName' => $routeName, 'breadcrumb' => $breadcrumb]);
        }
        return redirect()->route($routeName . '.index');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $data = $request->all();
        $data['password'] = Hash::make($data['password']);

        if (!$data['password']) {
            unset($data['password']);
        }

        Validator::make($data, [
            'nome' => 'required|string|max:255',
            'sobrenome' => 'required|string',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($id)],
            'password' => 'sometimes|required|string|min:6|confirmed',
            'cpf' => 'required|string|min:11',
            'telefone' => 'required|string|min:11',
            'data_nascimento' => 'required',
        ]);

        if ($this->model->update($data, $id)) {
            session()->flash('msg', trans('agendamento.successfully_edited_record'));
            session()->flash('status', 'success'); // success error notification
            return redirect()->back();
        } else {
            session()->flash('msg', trans('agendamento.error_editing_record'));
            session()->flash('status', 'error'); // success error notification
            return redirect()->back();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if ($this->model->delete($id)) {
            session()->flash('msg', trans('agendamento.registration_deleted_successfully'));
            session()->flash('status', 'error'); // success error notification
        } else {
            session()->flash('msg', trans('agendamento.error_deleting_record'));
            session()->flash('status', 'error'); // success error notification
        }

        $routeName = $this->route;
        return redirect()->route($routeName . '.index');
    }

    public function pesquisaTitulo()
    {
        $columnList = ['id' => '#', 'municipio' => trans('agendamento.municipio'), 'detentor' => trans('agendamento.name'),
            'cpf' => trans('agendamento.cpf'), 'titulo' => trans('agendamento.titulo'), 'nome_mae' => trans('agendamento.nome_mae'),
            'data' => trans('agendamento.data'), 'hora' => trans('agendamento.hora'), 'escritorio' => trans('agendamento.escritorio')];

        $page = trans('agendamento.agendamento_list');
        $page_create = trans('agendamento.agendamento');
        $routeName = $this->route;
        $list = $this->model->paginate($this->paginate);
        $breadcrumb = [
            (object) ['url' => route('home'), 'title' => trans('agendamento.home')],
            (object) ['url' => '', 'title' => trans('agendamento.list', ['page' => $page])],
        ];
        //dd("oi");

        // $listUsers = $this->model->all();
        // foreach ($listUsers as $user) {
        //     $documentosPessoa = DocumentoPessoa::where('cpf', $user->cpf)->with('pessoa')->first();
        //     //dd($documentosPessoa->cpf);
        //     if($user->cpf == $documentosPessoa->cpf){
        //        // dd("CPFS IGUAIS");
        //         echo $user->cpf;
        //         echo $documentosPessoa->cpf;
        //         dd("OK");
        //     }else{
        //         dd("CPFS DIFERENTES");
        //     }
            
        //     //dd($documentosPessoa);
        //     //$listUser = $this->model->findWhereLike(['cpf'], $user->cpf);
        //     //dd($user->cpf);
        // }

       
        // $documentosPessoa = DocumentoPessoa::where('cpf', $user->cpf)->with('pessoa')->get();
        // foreach ($documentosPessoa as $docPessoa) {
        //     echo $docPessoa;
        // }
        // //echo $listUser;

        

        return view('admin.agendamentos.agendamentoTitulo', ['page' => $page, 'columnList' => $columnList, 'routeName' => $routeName,
            'list' => $list, 'page_create' => $page_create, 'breadcrumb' => $breadcrumb]);

    }
}
